# exception_data.pm: details of the exception hierarchy used by Xapian.
package exception_data;
$copyright = <<'EOF';
/* Copyright (C) 2003,2004,2006,2007,2008,2009,2011,2015,2019 Olly Betts
 * Copyright (C) 2007 Richard Boulton
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */
EOF

use Exporter;
@ISA = qw(Exporter);
@EXPORT = qw(
    $copyright $generated_warning @baseclasses @classes %subclasses %classcode
);

$generated_warning =
"/* Warning: This file is generated by $0 - do not modify directly! */\n";

@baseclasses = ();
@classes = ();
%subclasses = ();
%classcode = ();

sub errorbaseclass {
    push @baseclasses, \@_;
    push @{$subclasses{$_[1]}}, $_[0];
}

sub errorclass {
    my $typecode = shift;
    my ($class, $parent) = @_;
    push @classes, \@_;
    push @{$subclasses{$parent}}, $class;
    $classcode{$class} = $typecode;
}

errorbaseclass('LogicError', 'Error',
	       'The base class for exceptions indicating errors in the program logic.',
	       <<'DOC');
A subclass of LogicError will be thrown if Xapian detects a violation
of a class invariant or a logical precondition or postcondition, etc.
DOC

errorclass(0, 'AssertionError', 'LogicError',
	   'AssertionError is thrown if a logical assertion inside Xapian fails.',
	   <<'DOC');
In a debug build of Xapian, a failed assertion in the core library code
will cause AssertionError to be thrown.

This represents a bug in Xapian (either an invariant, precondition, etc
has been violated, or the assertion is incorrect!)
DOC

errorclass(1, 'InvalidArgumentError', 'LogicError',
	   'InvalidArgumentError indicates an invalid parameter value was passed to the API.',
	   '');

errorclass(2, 'InvalidOperationError', 'LogicError',
	   'InvalidOperationError indicates the API was used in an invalid way.',
	   '');

errorclass(3, 'UnimplementedError', 'LogicError',
	   'UnimplementedError indicates an attempt to use an unimplemented feature.',
	   '');

# RuntimeError and subclasses:

errorbaseclass('RuntimeError', 'Error',
	       'The base class for exceptions indicating errors only detectable at runtime.',
	       <<'DOC');
A subclass of RuntimeError will be thrown if Xapian detects an error
which is exception derived from RuntimeError is thrown when an
error is caused by problems with the data or environment rather
than a programming mistake.
DOC

errorclass(4, 'DatabaseError', 'RuntimeError',
	   'DatabaseError indicates some sort of database related error.',
	   '');

errorclass(5, 'DatabaseCorruptError', 'DatabaseError',
	   'DatabaseCorruptError indicates database corruption was detected.',
	   '');

errorclass(6, 'DatabaseCreateError', 'DatabaseError',
	   'DatabaseCreateError indicates a failure to create a database.',
	   '');

errorclass(7, 'DatabaseLockError', 'DatabaseError',
	   'DatabaseLockError indicates failure to lock a database.',
	   '');

errorclass(8, 'DatabaseModifiedError', 'DatabaseError',
	   'DatabaseModifiedError indicates a database was modified.',
	   <<'DOC');
To recover after catching this error, you need to call
Xapian::Database::reopen() on the Database and repeat the operation
which failed.
DOC

errorclass(9, 'DatabaseOpeningError', 'DatabaseError',
	   'DatabaseOpeningError indicates failure to open a database.',
	   '');

errorclass(10, 'DatabaseVersionError', 'DatabaseOpeningError',
	   'DatabaseVersionError indicates that a database is in an unsupported format.',
	   <<'DOC');
From time to time, new versions of Xapian will require the database format
to be changed, to allow new information to be stored or new optimisations
to be performed.  Backwards compatibility will sometimes be maintained, so
that new versions of Xapian can open old databases, but in some cases
Xapian will be unable to open a database because it is in too old (or new)
a format.  This can be resolved either be upgrading or downgrading the
version of Xapian in use, or by rebuilding the database from scratch with
the current version of Xapian.
DOC

errorclass(11, 'DocNotFoundError', 'RuntimeError',
	   'Indicates an attempt to access a document not present in the database.',
	   '');

errorclass(12, 'FeatureUnavailableError', 'RuntimeError',
	   'Indicates an attempt to use a feature which is unavailable.',
	   <<'DOC');
Typically a feature is unavailable because it wasn't compiled in, or
because it requires other software or facilities which aren't available.
DOC

errorclass(13, 'InternalError', 'RuntimeError',
	   'InternalError indicates a runtime problem of some sort.',
	   '');

errorclass(14, 'NetworkError', 'RuntimeError',
	   'Indicates a problem communicating with a remote database.',
	   '');

errorclass(15, 'NetworkTimeoutError', 'NetworkError',
	   'Indicates a timeout expired while communicating with a remote database.',
	   '');

errorclass(16, 'QueryParserError', 'RuntimeError',
	   "Indicates a query string can't be parsed.",
	   '');

errorclass(17, 'SerialisationError', 'RuntimeError',
	   'Indicates an error in the std::string serialisation of an object.',
	   '');

errorclass(18, 'RangeError', 'RuntimeError',
	   'RangeError indicates an attempt to access outside the bounds of a container.',
	   '');

errorclass(19, 'WildcardError', 'RuntimeError',
	   'WildcardError indicates an error expanding a wildcarded query.',
	   '');

errorclass(20, 'DatabaseNotFoundError', 'DatabaseOpeningError',
	   'Indicates an attempt to access a database not present.',
	   '');

errorclass(21, 'DatabaseClosedError', 'DatabaseError',
	   'Indicates an attempt to access a closed database.',
	   '');

sub for_each_nothrow {
    my $func = shift @_;
    my $class = '';
    foreach my $header ('include/xapian.h', <include/xapian/*.h>) {
	local $/ = undef;
	open H, '<', $header or die $!;
	my $header_text = <H>;
	# Strip comments, which might contain text describing XAPIAN_NOTHROW().
	$header_text =~ s!/(?:/[^\n]*|\*.*?\*/)! !gs;
	for (split /\n/, $header_text) {
	    if (/^\s*class\s+XAPIAN_VISIBILITY_DEFAULT\s+(\w+)/) {
		$class = "$1::";
		&$func("Xapian::$class~$1");
		next;
	    }
	    if (/^[^#]*\bXAPIAN_NOTHROW\(([^~].*)\)/) {
		&$func("Xapian::$class$1");
	    }
	}
	close H;
    }
}

1;
